unit Buscador;

interface

uses classes, ComCtrls, Windows;

type
  TBuscador = class;

  TBusqueda = class(TThread)
  private
    FBuscador:    TBuscador;
    FRuta:        string;
    FCarpeta:     string;
    FSubcarpetas: boolean;
    FNodo:        TTreeNode;

    FResultado:   boolean;

    procedure OnEnd(Sender: TObject);

  public
    constructor Create(const ABuscador: TBuscador;
                       const ANodoPadre: TTreeNode;
                       const ARuta: string;
                       const ASubcarpetas: boolean); reintroduce; overload;
    destructor Destroy; override;

    procedure Execute; override;
  end;


  TBuscador = class(TObject)
  private
    FCriticalArbol: TRTLCriticalSection;
    FCriticalLista: TRTLCriticalSection;
    FCriticalHilos: TRTLCriticalSection;

    // configuracin de la bsqueda
    FLista:       TListView;
    FArbol:       TTreeView;
    FCarpeta:     string;
    FSubcarpetas: boolean;

    // datos estadsticos
    FMSecs:      LongWord;

    // evento al finalizar la bsqueda
    FOnEnd: TNotifyEvent;

    // lista interna de hilos
    FThreads: TList;

  public
    constructor Create;
    destructor Destroy; override;

    procedure AddThread(const Hilo: TThread);
    procedure DeleteThread(const Hilo: TThread);

    procedure Execute;
    procedure Pause(parar: boolean);

    property OnEnd: TNotifyEvent write FOnEnd;

    property Lista:       TListView read FLista       write FLista;
    property Arbol:       TTreeView read FArbol       write FArbol;
    property Carpeta:     string    read FCarpeta     write FCarpeta;
    property Subcarpetas: boolean   read FSubcarpetas write FSubcarpetas;
    property MSecs:       LongWord  read FMSecs;
  end;


implementation

uses SysUtils, Forms, FileCtrl;


//~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
constructor TBusqueda.Create(const ABuscador: TBuscador; const ANodoPadre: TTreeNode;
                             const ARuta: string; const ASubcarpetas: boolean);

  procedure SetCarpeta;
  var
    pi, pf, PRuta: PChar;
  begin
    PRuta := PChar(FRuta);
    pf    := StrRScan(PRuta, '\');

    if pf = nil then
      FCarpeta := FRuta

    else
    begin
      pf^ := #0;
      try

        pi := StrRScan(PRuta, '\');
        if pi = nil then
        begin
          FCarpeta := StrPas(PRuta);
        end
        else
        begin
          Inc(pi);
          FCarpeta := StrPas(pi);
        end;

      finally
        pf^ := '\';
      end;

    end;
  end;

begin
  inherited Create(true);

  FBuscador    := ABuscador;
  FSubcarpetas := ASubcarpetas;
  FRuta        := ARuta;
  SetCarpeta;

  Self.OnTerminate     := Self.OnEnd;
  Self.FreeOnTerminate := true;

  FBuscador.AddThread(Self);

  EnterCriticalSection(FBuscador.FCriticalArbol);
  FBuscador.FArbol.Items.BeginUpdate;
  try
    if ANodoPadre = nil then
      FNodo := FBuscador.FArbol.Items.Add(nil, 'Buscando en "' + FCarpeta + '"...')
    else
    begin
      FNodo := FBuscador.FArbol.Items.AddChild(ANodoPadre, 'Buscando en "' + FCarpeta + '"...');

      if not ANodoPadre.Expanded then
        ANodoPadre.Expand(false);
    end;

    FNodo.ImageIndex    := 0;
    FNodo.SelectedIndex := FNodo.ImageIndex;

    // cada vez que se crea un hilo, es que estamos en una nueva carpeta
  finally
    FBuscador.FArbol.Items.EndUpdate;
    LeaveCriticalSection(FBuscador.FCriticalArbol);
  end;

end;

//~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
destructor TBusqueda.Destroy;
begin
  FBuscador.DeleteThread(self);
  inherited;
end;

//~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
procedure TBusqueda.OnEnd(Sender: TObject);
begin

  if FResultado then
  begin
    FNodo.Text := FCarpeta + ': fichero(s) encontrado(s).';
    FNodo.ImageIndex := 2;
  end
  else
  begin
    FNodo.Text := FCarpeta + ': No contiene el fichero buscado.';
    FNodo.ImageIndex := 1;
  end;
  
end;

//~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
procedure TBusqueda.Execute;
var
    FindData:     WIN32_FIND_DATA;
    SearchHandle: THandle;
    Hay:          boolean;

    ruta:         array[0..MAX_PATH-1] of char;
    archivo:      array[0..MAX_PATH-1] of char;
    i:            integer;

    RutaArg:      array[0..MAX_PATH-1] of char;

    Busqueda :    TBusqueda;

    carpeta:      PChar;
    ultimo:       TListItem;
begin
  Hay     := false;
  carpeta := PChar(FRuta);

  //
  // busca en cada una de las subcarpetas (si procede)
  //
  if FSubcarpetas then
  begin
    ZeroMemory(@ruta, MAX_PATH);
    ZeroMemory(@archivo, MAX_PATH);

    //
    // Me quedo con la ruta slamente.
    //
    for i := StrLen(carpeta) - 1 downto 1 do
    begin

      if carpeta[i] = '\' then
      begin
        StrLCopy(ruta, carpeta, i + 1);
        StrCopy(archivo, PChar(Integer(@carpeta[i]) + 1));
        break;
      end;

    end;

    StrCat(ruta, '*.*');

    SearchHandle := FindFirstFile(ruta, FindData);

    if SearchHandle <> INVALID_HANDLE_VALUE then
    begin
      //
      // Se itera en la carpeta actual buscando subcarpetas
      //
      repeat
        // si es carpeta, pongo el progreso y creo nuevo hilo
        if (FindData.dwFileAttributes and FILE_ATTRIBUTE_DIRECTORY <> 0) and
           (FindData.cFileName[0] <> '.') then
        begin
          ZeroMemory(@RutaArg, MAX_PATH);

          StrLCopy(RutaArg, ruta, StrLen(ruta) - 3);
          StrCat(RutaArg, FindData.cFileName);
          StrCat(RutaArg, '\');
          StrCat(RutaArg, archivo);

          Busqueda := TBusqueda.Create(FBuscador, FNodo, RutaArg, true);

          Busqueda.Resume;

        end;

      until not FindNextFile(SearchHandle, FindData);

      //
      // error en algn paso de la bsqueda
      //
      if GetLastError <> ERROR_NO_MORE_FILES then
      begin
        Windows.FindClose(SearchHandle);
        FResultado := false;
        RaiseLastWin32Error;
      end;

      Windows.FindClose(SearchHandle);

    end;

  end;

  // busco el archivo en la carpeta actual
  SearchHandle := FindFirstFile(carpeta, FindData);

  if SearchHandle <> INVALID_HANDLE_VALUE then
  begin
    ultimo := nil;

    // Se itera en la carpeta actual
    repeat

      // si encuentro algo, inserto el resultado (sincronizado)
      if not ((FindData.dwFileAttributes and FILE_ATTRIBUTE_DIRECTORY <> 0) or
              (FindData.cFileName[0] = '.')) then
      begin
      
        EnterCriticalSection(FBuscador.FCriticalLista);
        try
          ultimo         := FBuscador.FLista.Items.Add;
          ultimo.Caption := ExtractFilePath(carpeta) + FindData.cFileName;
        finally
          LeaveCriticalSection(FBuscador.FCriticalLista);
        end;

        Hay := true;
      end;
    until not FindNextFile(SearchHandle, FindData);

    if hay and (ultimo <> nil) then
      FBuscador.FLista.Selected := ultimo;

    //
    // error en algn paso de la bsqueda
    //
    if GetLastError <> ERROR_NO_MORE_FILES then
    begin
      Windows.FindClose(SearchHandle);
      FResultado := false;
      RaiseLastWin32Error;
    end;

  end;

  FResultado := Hay;

end;


//~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
// Clase Buscador
//~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
constructor TBuscador.Create;
begin
  inherited Create;

  InitializeCriticalSection(FCriticalArbol);
  InitializeCriticalSection(FCriticalHilos);
  InitializeCriticalSection(FCriticalLista);

  FThreads := TList.Create;
end;

//~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
destructor TBuscador.Destroy;
begin
  DeleteCriticalSection(FCriticalArbol);
  DeleteCriticalSection(FCriticalHilos);
  DeleteCriticalSection(FCriticalLista);

  FThreads.Free;
end;

//~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
procedure TBuscador.AddThread(const Hilo: TThread);
begin

  EnterCriticalSection(FCriticalHilos);
  try
    if FThreads.IndexOf(Hilo) = -1 then
      FThreads.Add(Hilo);
  finally
    LeaveCriticalSection(FCriticalHilos);
  end;

end;

//~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
procedure TBuscador.DeleteThread(const Hilo: TThread);
var
  SinHilos: boolean;
  ind:      integer;
begin

  EnterCriticalSection(FCriticalHilos);
  try

    ind := FThreads.IndexOf(hilo);
    FThreads.Delete(ind);
    SinHilos := (FThreads.Count = 0);

  finally
    LeaveCriticalSection(FCriticalHilos);
  end;

  // si ya no quedan hilos, lanzo el evento en cuestin
  if SinHilos then
  begin
    FMSecs := GetTickCount - FMSecs;
    if Assigned(FOnEnd) then
      FOnEnd(Self);
  end;

end;

//~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
procedure TBuscador.Execute;
var
  busqueda: TBusqueda;
begin
  FMSecs   := 0;

  // creo un hilo y lo lanzo
  FMSecs := GetTickCount;
  busqueda := TBusqueda.Create(Self, nil, FCarpeta, FSubcarpetas);
  busqueda.Resume;
end;

procedure TBuscador.Pause(parar: boolean);
var
  i: integer;
begin

  if parar then
  begin
    for i := FThreads.Count - 1 downto 0 do
      TThread(FThreads[i]).Suspend;

    // activo el repintado
    FArbol.Items.EndUpdate;
  end
  else
    for i := FThreads.Count - 1 downto 0 do
      TThread(FThreads[i]).Resume;
end;



end.
